"""
Recursive Audio Metadata Tagger (URL Cover Art + Force Update)
---------------------------------------------------

Features:
- Recursively scans current directory and subdirectories
- Smart MP3 / WAV detection
- Auto-derives TITLE from audio filename
- User-specified Cover Art URL (Downloads & Embeds)
- Website + PayPal stored separately
- MP3 & WAV cover art embedding (APIC via ID3)
- FORCES update (overwrites existing tags)

Requirements:
    pip install mutagen requests
"""

import os
import re
import requests
from mutagen.mp3 import MP3
from mutagen.id3 import (
    ID3, TIT2, TPE1, TALB, TDRC,
    COMM, TCON, TXXX, APIC, WOAR, WXXX
)
from mutagen.wave import WAVE

AUDIO_EXTS = (".mp3", ".wav")

# ------------------------------------------------------------
# SMART TITLE FROM FILENAME
# ------------------------------------------------------------

def smart_title_from_filename(path):
    name = os.path.splitext(os.path.basename(path))[0]
    # Remove leading numbers/track counts (e.g. "01 - Title" -> "Title")
    name = re.sub(r"^[\[\(]?\d{1,3}[\]\)]?\s*[-_.]?\s*", "", name)
    name = re.sub(r"[_\-.]+", " ", name)

    junk = {
        "final", "mix", "master", "demo",
        "edit", "render", "export", "bounce"
    }

    words = [w for w in name.split() if w.lower() not in junk]
    title = " ".join(words).strip().title()
    return title or "Untitled"

# ------------------------------------------------------------
# COVER ART (VIA URL)
# ------------------------------------------------------------

def download_image_data(url):
    """Downloads image bytes from a URL."""
    if not url:
        return None, None
    
    try:
        print(f"    ⬇ Downloading art from: {url}")
        response = requests.get(url, timeout=10)
        response.raise_for_status()
        
        # Determine mime type from headers or extension
        content_type = response.headers.get('content-type', '')
        if 'image/png' in content_type:
            mime = 'image/png'
        elif 'image/jpeg' in content_type:
            mime = 'image/jpeg'
        else:
            # Fallback based on URL string
            if url.lower().endswith('.png'):
                mime = 'image/png'
            else:
                mime = 'image/jpeg'
                
        return response.content, mime
    except Exception as e:
        print(f"    ⚠ Error downloading image: {e}")
        return None, None

def embed_cover_art(audio, image_data, mime_type):
    """Embeds cover art into ID3 tags (works for MP3 and WAV with ID3 chunk)."""
    if not image_data:
        return

    # 'audio.tags' is the ID3 container for both MP3 and WAVE in Mutagen
    audio.tags.delall("APIC")
    audio.tags.add(APIC(
        encoding=3,
        mime=mime_type,
        type=3,  # Front cover
        desc="Cover",
        data=image_data
    ))

# ------------------------------------------------------------
# USER INPUT
# ------------------------------------------------------------

def prompt_metadata():
    print("\nShared metadata (TITLE auto-generated):\n")
    
    data = {
        "artist": input("Artist: ").strip(),
        "album": input("Album: ").strip(),
        "year": input("Year (YYYY): ").strip(),
        "genre": input("Genre: ").strip(),
        "website": input("Website URL: ").strip(),
        "paypal": input("PayPal Donation URL: ").strip(),
        "comment": input("License / Comment text: ").strip(),
        "cover_url": input("Cover Art URL (e.g. https://site.com/art.jpg): ").strip(),
        "tagged_flag": "Tagged by batch_audio_metadata_recursive.py"
    }
    
    # Pre-download image once to save bandwidth/time
    img_data, mime = None, None
    if data["cover_url"]:
        img_data, mime = download_image_data(data["cover_url"])
    
    return data, img_data, mime

# ------------------------------------------------------------
# GENERIC ID3 TAGGING (MP3 & WAV)
# ------------------------------------------------------------

def apply_id3_tags(audio, path, meta, img_data, mime):
    """Applies ID3 tags to an audio object (MP3 or WAVE)."""
    
    # 1. Title (derived from audio filename)
    audio.tags.add(TIT2(encoding=3, text=smart_title_from_filename(path)))

    # 2. Standard Fields
    if meta["artist"]:
        audio.tags.add(TPE1(encoding=3, text=meta["artist"]))
    if meta["album"]:
        audio.tags.add(TALB(encoding=3, text=meta["album"]))
    if meta["year"]:
        audio.tags.add(TDRC(encoding=3, text=meta["year"]))
    if meta["genre"]:
        audio.tags.add(TCON(encoding=3, text=meta["genre"]))

    # 3. URLs
    if meta["website"]:
        audio.tags.add(WOAR(encoding=3, url=meta["website"]))
    
    if meta["paypal"]:
        audio.tags.add(WXXX(encoding=3, desc="PayPal", url=meta["paypal"]))
        audio.tags.add(TXXX(encoding=3, desc="PayPal", text=meta["paypal"]))

    # 4. Comments
    if meta["comment"]:
        audio.tags.add(COMM(
            encoding=3,
            lang="eng",
            desc="Comment",
            text=meta["comment"]
        ))

    # 5. Cover Art (Embedded from memory)
    if img_data:
        try:
            embed_cover_art(audio, img_data, mime)
        except Exception as e:
            print(f"    ⚠ Could not embed art: {e}")

    # 6. Mark as Tagged
    audio.tags.add(TXXX(encoding=3, desc="Tagged", text=meta["tagged_flag"]))
    
    audio.save()
    return True

# ------------------------------------------------------------
# FILE HANDLERS
# ------------------------------------------------------------

def tag_mp3(path, meta, img_data, mime):
    audio = MP3(path, ID3=ID3)
    if audio.tags is None:
        audio.add_tags()
    return apply_id3_tags(audio, path, meta, img_data, mime)

def tag_wav(path, meta, img_data, mime):
    audio = WAVE(path)
    if audio.tags is None:
        audio.add_tags()
    return apply_id3_tags(audio, path, meta, img_data, mime)

# ------------------------------------------------------------
# MAIN
# ------------------------------------------------------------

def main():
    # Unpack tuple returned by prompt_metadata
    meta, img_data, img_mime = prompt_metadata()
    tagged = 0

    print("\nScanning audio files...\n")

    for root, _, files in os.walk("."):
        for f in files:
            if not f.lower().endswith(AUDIO_EXTS):
                continue

            path = os.path.join(root, f)
            try:
                # We removed the skip logic. This always processes the file.
                if f.lower().endswith(".mp3"):
                    tag_mp3(path, meta, img_data, img_mime)
                elif f.lower().endswith(".wav"):
                    tag_wav(path, meta, img_data, img_mime)

                # Always successful unless Exception raised
                tagged += 1
                print(f"✔ Tagged: {path}")

            except Exception as e:
                print(f"⚠ Error: {path} → {e}")

    print("\n✅ Done")
    print(f"Files Processed: {tagged}")

if __name__ == "__main__":
    main()